// SPDX-FileCopyrightText: 2023 g10 code Gmbh
// SPDX-Contributor: Carl Schwan <carl.schwan@gnupg.com>
// SPDX-License-Identifier: GPL-2.0-or-later

#include <QApplication>
#include <QCommandLineParser>
#include <QElapsedTimer>
#include <QFile>
#include <QHttpServer>
#include <QHttpServerResponse>
#include <QJsonArray>
#include <QJsonDocument>
#include <QJsonObject>
#include <QNetworkReply>
#include <QPointer>
#include <QTimer>
#include <QUuid>

#include <Libkleo/KeyCache>

#include <KAboutData>
#include <KJob>
#include <KLocalizedString>

#include "../common/log.h"
#include "config.h"
#include "firsttimedialog.h"
#include "gpgol_client_debug.h"
#include "gpgolweb_version.h"
#include "utils/kuniqueservice.h"

using namespace Qt::Literals::StringLiterals;
using namespace std::chrono;

#ifdef Q_OS_WINDOWS
#include <windows.h>
#endif

#define STARTUP_TIMING qCDebug(GPGOL_CLIENT_LOG) << "Startup timing:" << startupTimer.elapsed() << "ms:"
#define STARTUP_TRACE qCDebug(GPGOL_CLIENT_LOG) << "Startup timing:" << startupTimer.elapsed() << "ms:" << SRCNAME << __func__ << __LINE__;

static void setupLogging(std::shared_ptr<Kleo::Log> log)
{
    const QByteArray dirNative = qgetenv("GPGOL_CLIENT_LOGDIR");
    if (dirNative.isEmpty()) {
        return;
    }
    log->setOutputDirectory(QFile::decodeName(dirNative));
    qInstallMessageHandler(Kleo::Log::messageHandler);
}

int main(int argc, char *argv[])
{
#ifdef Q_OS_WINDOWS
    if (AttachConsole(ATTACH_PARENT_PROCESS)) {
        freopen("CONOUT$", "w", stdout);
        freopen("CONOUT$", "w", stderr);
    }
#endif

    QApplication app(argc, argv);
    app.setQuitOnLastWindowClosed(false);

    KLocalizedString::setApplicationDomain(QByteArrayLiteral("gpgol-js-native"));

    KAboutData about(QStringLiteral("gpgol-client"),
                     i18nc("@title:window", "GnuPG Outlook Add-in"),
                     QStringLiteral(GPGOLWEB_VERSION_STRING),
                     i18nc("@info", "GPG Outlook add-in"),
                     KAboutLicense::GPL,
                     i18nc("@info:credit", "© 2023-2024 g10 Code GmbH"));

    about.setDesktopFileName(u"com.gnupg.gpgolweb"_s);
    about.setProgramLogo(QIcon::fromTheme(u"com.gnupg.gpgolweb"_s));

    about.addAuthor(i18nc("@info:credit", "Carl Schwan"), i18nc("@info:credit", "Maintainer"), u"carl.schwan@gnupg.com"_s, u"https://carlschwan.eu"_s);

    about.setTranslator(i18nc("NAME OF TRANSLATORS", "Your names"), i18nc("EMAIL OF TRANSLATORS", "Your emails"));

    about.setOrganizationDomain("gnupg.com");
    about.setBugAddress("https://dev.gnupg.org/maniphest/task/edit/form/1/");

    QCommandLineParser parser;

    KAboutData::setApplicationData(about);

    about.setupCommandLine(&parser);
    parser.process(app);
    about.processCommandLine(&parser);

    QElapsedTimer startupTimer;
    startupTimer.start();

    STARTUP_TIMING << "Application created";
    /* Create the unique service ASAP to prevent double starts if
     * the application is started twice very quickly. */
    KUniqueService service;

    QObject::connect(&service, &KUniqueService::activateRequested, &service, [&service](const QStringList &arguments, const QString &workingDirectory) {
        Q_UNUSED(arguments);
        Q_UNUSED(workingDirectory);
        service.setExitValue(0);
    });

    auto log = Kleo::Log::mutableInstance();
    setupLogging(log);

    STARTUP_TIMING << "Service created";

    QPointer<FirstTimeDialog> launcher = new FirstTimeDialog;

    if (Config::self()->showLauncher()) {
        launcher->show();
    }

    STARTUP_TIMING << "KeyCache creation";
    auto keyCache = Kleo::KeyCache::mutableInstance();
    keyCache->startKeyListing();

    return app.exec();
}
