// SPDX-FileCopyrightText: 2025 g10 code GmbH
// SPDX-FileContributor: Carl Schwan <carl.schwan@gnupg.com>
// SPDX-License-Identifier: GPL-2.0-or-later

#include "reencryptjob.h"
#include "choosekeydialog.h"
#include "ews/ewscopyitemrequest.h"
#include "ews/ewscreatefolderrequest.h"
#include "ews/ewscreateitemrequest.h"
#include "ews/ewsfinditemrequest.h"
#include "ews/ewsgetfoldercontentrequest.h"
#include "ews/ewsgetfolderrequest.h"
#include "ews/ewsgetitemrequest.h"
#include "ews/ewsitem.h"
#include "ews/ewsupdateitemrequest.h"

#include <QGpgME/DecryptJob>
#include <QGpgME/EncryptJob>
#include <QGpgME/Protocol>
#include <gpgme++/context.h>
#include <gpgme++/decryptionresult.h>
#include <gpgme++/encryptionresult.h>

#include <QFile>
#include <QLabel>
#include <QListView>
#include <QLocale>
#include <QPlainTextEdit>
#include <QSaveFile>
#include <QStandardItemModel>
#include <QStyledItemDelegate>
#include <QTimer>
#include <QVBoxLayout>

#include <KLocalizedString>
#include <KMessageBox>
#include <KMime/Message>
#include <Libkleo/Formatting>
#include <Libkleo/KeyCache>

using namespace Qt::StringLiterals;
using namespace std::chrono_literals;

ReencryptJob::ReencryptJob(WebsocketClient *websocketClient, const EwsId &folderId, const EwsClient &client)
    : KJob()
    , m_itemModel(new QStandardItemModel(this))
    , m_websocketClient(websocketClient)
    , m_folderId(folderId)
    , m_ewsClient(client)
{
    setAutoDelete(false);
}

void ReencryptJob::start()
{
    auto request = new EwsGetFolderRequest(m_ewsClient, this);
    request->setFolderIds({m_folderId});

    connect(request, &EwsGetFolderRequest::finished, this, [this, request](KJob *) {
        if (request->error() != KJob::NoError) {
            setError(request->error());
            setErrorText(request->errorText());
            emitResult();
            return;
        }

        const auto &responses = request->responses();
        if (responses.isEmpty()) {
            setError(KJob::UserDefinedError);
            setErrorText(u"Empty response from EwsGetFolderRequest"_s);
            emitResult();
            return;
        }

        m_folder = responses.at(0).folder();

        auto dialog = new ChooseKeyDialog(m_folder);
        dialog->setAttribute(Qt::WA_DeleteOnClose);
        connect(dialog, &QDialog::accepted, this, [this, dialog] {
            m_currentKeys = dialog->currentKeys();
            m_unencryptedMode = dialog->unencryptedMode();

            auto dialog = new QDialog;
            auto layout = new QVBoxLayout(dialog);
            m_title = new QLabel(i18n("Creating folder"));
            layout->addWidget(m_title);

            auto listView = new QListView;
            listView->setModel(m_itemModel);
            listView->setItemDelegate(new QStyledItemDelegate);
            layout->addWidget(listView);

            auto debugLogLabel = new QLabel(i18nc("@label", "Debug logs:"));
            layout->addWidget(debugLogLabel);
            m_debugLog = new QPlainTextEdit;
            layout->addWidget(m_debugLog);

            dialog->show();
            dialog->setAttribute(Qt::WA_DeleteOnClose);
            connect(dialog, &QDialog::destroyed, this, [this]() {
                if (this->m_finished) {
                    this->deleteLater();
                    return;
                }
                this->setAutoDelete(true);
            });

            createNewFolder();
        });
        connect(dialog, &QDialog::rejected, this, [this] {
            emitResult();
        });
        dialog->show();
    });
    request->start();
}

void ReencryptJob::createNewFolder(int index)
{
    EwsFolder newFolder;
    newFolder.setType(EwsFolderTypeMail);
    newFolder.setField(
        EwsFolderFieldDisplayName,
        u"%1 - reencrypted%2"_s.arg(m_folder[EwsItemFields::EwsFolderFieldDisplayName].toString(), index == 0 ? QString{} : u" %1"_s.arg(index)));

    auto request = new EwsCreateFolderRequest(m_ewsClient, this);
    request->setFolders({newFolder});
    if (m_folder.hasField(EwsItemFieldParentFolderId)) {
        request->setParentFolderId(m_folder[EwsItemFieldParentFolderId].value<EwsId>());
    } else {
        request->setParentFolderId(EwsId{EwsDIdMsgFolderRoot});
    }
    connect(request, &EwsCreateFolderRequest::finished, this, [this, index, request](KJob *) {
        if (request->error() != KJob::NoError) {
            setError(request->error());
            setErrorText(request->errorText());

            qWarning() << "folder not created" << request->errorText();
            emitResult();
            return;
        }

        if (request->responses().at(0).folderId().id().isNull()) {
            createNewFolder(index + 1);
            return;
        }

        m_newFolderId = request->responses().at(0).folderId();

        m_getFolderContentRequest = new EwsGetFolderContentRequest(m_folderId, m_ewsClient);
        connect(m_getFolderContentRequest, &EwsGetFolderContentRequest::fetchedItems, this, [this](const QList<EwsItem> &items) {
            for (const auto &item : items) {
                m_itemsToReencrypt.enqueue(item);
            }

            m_title->setText(i18nc("@title:dialog",
                                   "Fetched %1 and processed %2 from %3 emails",
                                   m_getFolderContentRequest->offset(),
                                   m_countReencrypted,
                                   m_getFolderContentRequest->totalItems()));

            reencrypt();
        });
        connect(m_getFolderContentRequest, &EwsGetFolderRequest::result, m_getFolderContentRequest, [this](KJob *) {
            m_finished = true;
        });
        m_getFolderContentRequest->start();

    });
    request->start();
}

template<typename T>
const T *findHeader(KMime::Content *content)
{
    auto header = content->header<T>();
    if (header || !content->parent()) {
        return header;
    }
    return findHeader<T>(content->parent());
}

void ReencryptJob::reencrypt()
{
    if (m_reencrypting || m_itemsToReencrypt.isEmpty()) {
        if (m_finished) {
            emitResult();
        }
        return;
    }

    m_countReencrypted++;
    m_title->setText(i18nc("@title:dialog",
                           "Fetched %1 and processed %2 from %3 emails",
                           m_getFolderContentRequest->offset(),
                           m_countReencrypted,
                           m_getFolderContentRequest->totalItems()));
    const auto item = m_itemsToReencrypt.dequeue();
    m_reencrypting = true;
    const auto mimeContent = item[EwsItemFieldMimeContent].toString().toUtf8();
    if (mimeContent.isEmpty()) {
        m_reencrypting = false;
        reencrypt();
        return;
    }

    const auto mailData = KMime::CRLFtoLF(mimeContent);
    const auto msg = KMime::Message::Ptr(new KMime::Message);
    msg->setContent(mailData);
    msg->parse();

    auto subJobs = std::make_shared<ReencryptSubJobs>();

    auto subject = findHeader<KMime::Headers::Subject>(msg.get());

    subJobs->modelItem = new QStandardItem(subject ? subject->asUnicodeString() : u"no subject"_s);
    subJobs->modelItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
    subJobs->message = msg;
    subJobs->item = item;

    m_itemModel->appendRow(subJobs->modelItem);
    parseParts(item, msg.get(), subJobs);

    if (!subJobs->isEncrypted()) {
        saveCopy(item[EwsItemFieldItemId].value<EwsId>(), subJobs->modelItem);
    } else {
        QTimer::singleShot(10s, [this, subJobs]() {
            // Check if reencryption was finished
            if (!subJobs->encryptJobs.isEmpty()) {
                subJobs->modelItem->setIcon(QIcon::fromTheme(u"data-error"_s));
                subJobs->modelItem->setToolTip(i18n("Decryption timeout"));
                m_debugLog->appendPlainText(i18n("Unable to reencrypt \"%1\". Timing out in GPG. Treating as unencrypted", subJobs->modelItem->text()));

                saveCopy(subJobs->item[EwsItemFieldItemId].value<EwsId>(), subJobs->modelItem);
            }
        });
    }
}

void ReencryptJob::saveCopy(const EwsId &itemId, QStandardItem *index)
{
    if (m_unencryptedMode == UnencryptedMode::Copy) {
        auto copyItemRequest = new EwsCopyItemRequest(m_ewsClient, this);
        copyItemRequest->setItemIds({itemId});
        copyItemRequest->setDestinationFolderId(m_newFolderId);
        connect(copyItemRequest, &EwsCopyItemRequest::finished, this, [this, index, copyItemRequest]() {
            m_reencrypting = false;
            reencrypt();
            if (copyItemRequest->error() != KJob::NoError) {
                setError(KJob::UserDefinedError);
                setErrorText(errorText() + copyItemRequest->errorString() + u'\n');
                index->setIcon(QIcon::fromTheme(u"data-error"_s));
                m_debugLog->appendPlainText(copyItemRequest->errorString());
                return;
            }
            if (index->icon().name() != "data-error"_L1) {
                index->setIcon(QIcon::fromTheme(u"data-success"_s));
            }
        });
        copyItemRequest->start();
    } else {
        if (index->icon().name() != "data-error"_L1) {
            index->setIcon(QIcon::fromTheme(u"data-information"_s));
            index->setToolTip(i18nc("Skipped processing email", "Skipped processing email"));
        }
        m_reencrypting = false;
        reencrypt();
    }
}

void ReencryptJob::parseParts(const EwsItem &item, KMime::Content *content, std::shared_ptr<ReencryptSubJobs> subJobs)
{
    if (const auto contentType = content->contentType(); contentType) {
        const auto mimeType = contentType->mimeType();
        if (mimeType == "application/pgp-encrypted") {
            const auto parent = content->parent();
            if (!parent) {
                return;
            }

            const auto siblings = parent->contents();
            for (const auto sibling : siblings) {
                if (const auto contentType = sibling->contentType(); contentType) {
                    const auto mimeType = contentType->mimeType();
                    if (mimeType == "application/octet-stream") {
                        const auto encryptedContent = sibling->decodedContent();

                        auto job = QGpgME::openpgp()->encryptJob(true, true);
#ifdef GPGME2
                        job->setEncryptionFlags(GpgME::Context::AddRecipient);
#endif

                        job->setInputEncoding(GpgME::Data::ArmorEncoding);
                        subJobs->modelItem->setIcon(QIcon::fromTheme(u"view-refresh-symbolic"_s));

                        qWarning() << "Started reencrypting" << subJobs->modelItem->text() << "size" << encryptedContent.size();

                        connect(job,
                                &QGpgME::EncryptJob::result,
                                this,
                                [this, sibling, job, subJobs](const GpgME::EncryptionResult &result,
                                                              const QByteArray &cipherText,
                                                              const QString &,
                                                              const GpgME::Error &) {
                                    subJobs->encryptJobs.removeAll(job);

                                    if (result.error()) {
                                        // TODO: If we ever want to process things we can't decrypt other than skip item
                                        // code is needed here somehow to deal with it
                                        qWarning() << "Finished reencrypting" << subJobs->modelItem->text() << "with error"
                                                   << Kleo::Formatting::errorAsString(result.error());
                                        setError(KJob::UserDefinedError);
                                        setErrorText(errorText() + Kleo::Formatting::errorAsString(result.error()) + u'\n');
                                        subJobs->modelItem->setIcon(QIcon::fromTheme(u"data-error"_s));
                                        subJobs->modelItem->setToolTip(Kleo::Formatting::errorAsString(result.error()));
                                        m_debugLog->appendPlainText(Kleo::Formatting::errorAsString(result.error()));
                                        if (subJobs->encryptJobs.isEmpty()) {
                                            m_reencrypting = false;
                                            reencrypt();
                                        }
                                        return;
                                    }

                                    qWarning() << "Finished reencrypting" << subJobs->modelItem->text();

                                    sibling->setBody(cipherText);

                                    if (subJobs->encryptJobs.isEmpty()) {
                                        subJobs->message->assemble();
                                        const auto newMessage = subJobs->message->encodedContent();
                                        saveAsNew(subJobs->item[EwsItemFieldItemId].value<EwsId>(), newMessage, subJobs->modelItem);
                                    }
                                });

                        auto error = job->start(m_currentKeys, encryptedContent);
                        if (error) {
                            qWarning() << "Coult not start reencrypting" << subJobs->modelItem->text() << "with error"
                                       << Kleo::Formatting::errorAsString(error);
                            setError(KJob::UserDefinedError);
                            setErrorText(errorText() + Kleo::Formatting::errorAsString(error) + u'\n');
                            subJobs->modelItem->setIcon(QIcon::fromTheme(u"data-error"_s));
                            m_debugLog->appendPlainText(Kleo::Formatting::errorAsString(error));
                            continue;
                        }

                        subJobs->modelItem->setIcon(QIcon::fromTheme(u"cloud-upload-symbolic"_s));
                        subJobs->encryptJobs.append(job);
                    }
                }
            }
        }
    }

    for (const auto &content : content->contents()) {
        parseParts(item, content, subJobs);
    }
}

void ReencryptJob::saveAsNew(const EwsId &itemId, const QByteArray &copy, QStandardItem *index)
{
    auto copyItemRequest = new EwsCopyItemRequest(m_ewsClient, this);
    copyItemRequest->setItemIds({itemId});
    copyItemRequest->setDestinationFolderId(m_newFolderId);
    connect(copyItemRequest, &EwsCopyItemRequest::finished, this, [this, copy, index, copyItemRequest]() {
        if (copyItemRequest->error() != KJob::NoError) {
            setError(KJob::UserDefinedError);
            setErrorText(errorText() + copyItemRequest->errorString() + u'\n');
            index->setIcon(QIcon::fromTheme(u"data-error"_s));
            m_debugLog->appendPlainText(copyItemRequest->errorString());
            m_reencrypting = false;
            reencrypt();
            return;
        }

        auto updateItemRequest = new EwsUpdateItemRequest(m_ewsClient, this);
        EwsUpdateItemRequest::ItemChange itemChange(copyItemRequest->responses().at(0).itemId(), EwsItemTypeMessage);
        EwsPropertyField field(u"item:MimeContent"_s);
        itemChange.addUpdate(new EwsUpdateItemRequest::SetUpdate(field, QString::fromUtf8(copy.toBase64())));
        updateItemRequest->addItemChange(itemChange);
        updateItemRequest->setSavedFolderId(m_folderId);
        updateItemRequest->setConflictResolution(EwsConflictResolution::EwsResolAlwaysOverwrite);

        connect(updateItemRequest, &EwsUpdateItemRequest::finished, this, [this, index](KJob *job) {
            if (job->error()) {
                setError(KJob::UserDefinedError);
                setErrorText(errorText() + job->errorText() + u'\n');
                index->setIcon(QIcon::fromTheme(u"data-error"_s));
                index->setText(index->text() + u' ' + job->errorText());
                m_debugLog->appendPlainText(job->errorText());
                m_reencrypting = false;
                reencrypt();
                emitResult();
                return;
            }

            index->setIcon(QIcon::fromTheme(u"data-success"_s));

            m_reencrypting = false;
            reencrypt();
        });
        updateItemRequest->start();
    });

    copyItemRequest->start();
}
