// SPDX-FileCopyrightText: 2023 g10 code GmbH
// SPDX-Contributor: Carl Schwan <carl.schwan@gnupg.com>
// SPDX-License-Identifier: GPL-2.0-or-later

#pragma once

#include <QByteArray>
#include <QList>
#include <QObject>
#include <QObjectCleanupHandler>
#include <QSslError>

QT_FORWARD_DECLARE_CLASS(QWebSocketServer)

class QHttpServer;
class QSslServer;
class QWebSocket;

struct NativeClient {
    QWebSocket *socket;
    QString name;
    QString id;

    bool isValid() const
    {
        return socket != nullptr && !name.isEmpty() && !id.isEmpty();
    }

    bool operator==(const NativeClient &c) const
    {
        return socket == c.socket;
    }
};

class WebServer : public QObject
{
    Q_OBJECT
public:
    explicit WebServer(QObject *parent = nullptr);
    ~WebServer();

    /// Start web server.
    bool run();

    /// is a valid WebServer instance
    bool isValid() const;

    bool sendMessageToWebClient(const QString &email, const QByteArray &payload);
    bool sendMessageToNativeClient(const QString &email, const QJsonObject &obj, bool verify = true);

private Q_SLOTS:
    void onNewConnection();
    void processTextMessage(QString message);
    void processBinaryMessage(QByteArray message);
    void socketDisconnected();

private:
    enum SpecialValues {
        Port = 5656,
    };

    void processCommand(const QJsonObject &object, QWebSocket *socket);

    QHttpServer *const m_httpServer;
    std::unique_ptr<QSslServer> m_tcpserver;
    QObjectCleanupHandler m_clients;
    QHash<QString, QWebSocket *> m_webClientsMappingToEmail;

    QHash<QString, NativeClient> m_nativeClientsMappingToEmail;
};

inline size_t qHash(const NativeClient &client, size_t seed) noexcept
{
    return qHash(client.socket, seed);
}
